/*
 *          __        ___
 *    _____/ /___  __/ (_)____
 *   / ___/ __/ / / / / / ___/
 *  (__  ) /_/ /_/ / / (__  )
 * /____/\__/\__, /_/_/____/
 *          /____/
 *
 * stylis is a feature-rich css preprocessor
 *
 * @licence MIT
 */
(function (factory) {
	if (typeof exports === 'object' && typeof module !== 'undefined') {
		module.exports = factory(global);
	}
	else if (typeof define === 'function' && define.amd) {
		define(factory(window));
	}
	else {
		window.stylis = factory(window);
	}
}(function (window) {


	'use strict';


	// plugins
	var plugins = [];

	// regular expressions
	var andPattern = /&/g;
	var andSpacePattern = / +&/;
	var splitPattern = /,\n/g;
	var globalPattern = /:global\(%?((?:[^\(\)\[\]]*|\[.*\]|\([^\(\)]*\))*)\)/g;
	var globalsPattern = /(?:&| ):global\(%?((?:[^\(\)\[\]]*|\[.*\]|\([^\(\)]*\))*)\)/g;
	var hostPattern = /:host\((.*)\)/g;
	var hostContextPattern = /:host-context\((.*)\)/g;
	var newLinePattern = /\n/g;
	var placeholderPattern = /::place/g;
	var colonPattern = /: +/g;
	var animationPattern = /[ .#~+><\d]+/g;
	var transformPattern = / *(transform)/g;

	// prefixes
	var moz = '-moz-';
	var ms = '-ms-';
	var webkit = '-webkit-';


	/**
	 * css preprocessor
	 *
	 * @param  {String}   selector   - i.e `.class` or `#id` or `[attr=id]`
	 * @param  {String}   styles     - css string
	 * @param  {Boolean=} animations - prefix animations and keyframes, true by default
	 * @param  {Boolean=} compact    - enable additional shadow dom features(:host, :host-context)
	 * @param  {Function|Array} middlewares
	 * @return {string}
	 */
	function stylis (selector, styles, animations, compact, middlewares) {
		selector += '';

		var middleware = middlewares;
		var prefix = '';
		var namespace = '';
		var char;
		var attr;
		var animns;

		var type = selector.charCodeAt(0);

		// ` selector` -> `selector`
		if (type < 33) {
			type = (selector = selector.trim()).charCodeAt(0);
		}

		switch (type) {
			// `#` `.` id and class selectors
			case 35:
			case 46: {
				namespace = (prefix = selector).substring(1);
				break;
			}
			// [ attr selector
			case 91: {
				// `[data-id=namespace]` -> ['data-id', 'namespace']
				attr = selector.substring(1, selector.length - 1).split('=');
				char = (namespace = attr[1] || attr[0]).charCodeAt(0);

				// [data-id="namespace"]/[data-id='namespace']
				// --> "namespace"/'namspace' --> namespace
				if (char === 34 || char === 39) {
					namespace = namespace.substring(1, namespace.length - 1);
				}

				prefix = '['+ attr[0] + (attr.length > 1 ? ('="' + namespace +'"]') : ']')
				break;
			}
			// element selector
			default: {
				namespace = prefix = selector;
			}
		}

		// reset type signature
		type = 0;

		// animation and keyframe namespace
		if (animations === true || animations === undefined || animations === null) {
			animations = true;
			animns = namespace.replace(animationPattern, '-');
		}
		else {
			animns = '';
			animations = false;
		}

		// middleware
		var has;
		var uses = middleware !== void 0 && middleware !== null;
		var length = plugins.length;

		if (uses === true) {
			has = (typeof middleware).charCodeAt(0);

			// o, array
			if (has === 111) {
				use(middleware);
			}
			// f, function
			else if (has === 102) {
				plugins[length++] = middleware;
			}
			else {
				uses = false;
			}
		}

		if (length !== 0) {
			middleware = length === 1 ? plugins[0] : proxy;
			uses = true;
		}

		// declare
		var character;
		var colon;
		var inner;
		var selectors;
		var build;
		var temp;
		var prev;
		var indexOf;
		var first;
		var second;
		var third;
		var sel;
		var blob;
		var nest;
		var str;
		var media;

		// buffers
		var buff = '';
		var blck = '';
		var flat = '';

		// character code
		var code = 0;
		var nextcode;

		// context signatures
		var medias = 0;
		var special = 0;
		var close = 0;
		var closed = 0;
		var nested = 0;
		var func = 0;
		var strings = 0;
		var globs = 0;
		var isplace = 0;
		var join = 0;

		// context(flat) signatures
		var levels = 0;

		// comments
		var comment = 0;
		var comblck = 0;
		var comline = 0;

		// pre-process
		if (uses === true) {
			temp = middleware(0, styles, line, column, prefix, 0);

			if (temp != null) {
				styles = temp;
			}

			str = '';
		}

		// positions
		var caret = 0;
		var depth = 0;
		var column = 0;
		var line = 1;
		var eof = styles.length;

		// compiled output
		var output = '';

		// parse + compile
		while (caret < eof) {
			code = styles.charCodeAt(caret);

			// {, }, ; characters, parse line by line
			if (
				(medias === 1 && caret === eof - 1) ||
				(
					strings === 0 && func === 0 && comment === 0 &&
					(
						// {, }, ;
						(code === 123 || code === 125 || code === 59)
						||
						// eof buffer
						(caret === eof - 1 && buff.length !== 0)
					)
				)
			) {
				buff += styles.charAt(caret);

				// middleware, selector/property context, }
				if (uses === true && code !== 125) {
					// { pre-processed selector context
					if (code === 123) {
						temp = middleware(
							1,
							buff.substring(0, buff.length - 1).trim(),
							line,
							column,
							prefix,
							output.length
						);
					}
					// ; property context
					else {
						temp = middleware(2, buff, line, column, prefix, output.length);
					}

					if (temp != null) {
						buff = code === 123 ? temp + ' {' : temp;
					}
				}

				first = buff.charCodeAt(0);

				// only trim when the first character is a space ` `
				if (first === 32) {
					first = (buff = buff.trim()).charCodeAt(0);
				}

				second = buff.charCodeAt(1);
				third = buff.charCodeAt(2);

				// @, special block
				if (first === 64) {
					// push flat css
					if (levels === 1 && flat.length !== 0) {
						levels = 0;
						flat = prefix + ' {' + flat + '}';

						// middleware, flat context
						if (uses === true) {
							temp = middleware(4, flat, line, column, prefix, output.length);

							if (temp != null) {
								flat = temp;
							}
						}

						output += flat;
						flat = '';
					}

					// ;
					if (code !== 59) {
						// @keyframe, `k`
						if (second === 107) {
							blob = buff.substring(1, 11) + animns + buff.substring(11);
							buff = '@' + webkit + blob;
							type = 1;
						}
						// @media `m`, `e` characters,
						// @supports `s` `u` characters,
						// @global `g` character
						else if (
							(second === 109 && third === 101) ||
							(second === 115 && third === 117) ||
							(second === 103)
						) {
							caret++;
							column++;

							if (media === undefined) {
								media = '';
							}

							inner = '';

							// keep track of opening `{` and `}` occurrences
							closed = 1;

							// travel to the end of the block
							while (caret < eof) {
								char = styles.charCodeAt(caret);

								// {, }, nested blocks may have nested blocks
								if (char === 123) {
									closed++;
								}
								else if (char === 125) {
									closed--;
								}

								// break when the nested block has ended
								if (closed === 0) {
									caret++;
									break;
								}

								// build content of nested block
								inner += styles.charAt(caret++);

								// move column and line position
								column = (char === 13 || char === 10) ? (line++, 0) : column + 1;
							}

							selector = depth === 0 ? prefix : prev.substring(0, prev.length-1).replace(newLinePattern, '').trim();

							// build block
							media += (buff + stylis(
								selector,
								inner.trim(),
								animations,
								compact,
								null
							).trim() + '}');

							// middleware, block context
							if (uses === true) {
								temp = middleware(3, media, line, column, prefix, output.length);

								if (temp != null) {
									media = temp;
								}
							}

							buff = '';
							medias = 1;

							if (caret === eof) {
								eof++;
							}

							continue;
						}
						// unknown
						else {
							type = 6;
						}
					}

					// flag special, i.e @keyframes, @font-face ...
					if (code !== 59 && second !== 105) {
						// k, m
						if (second !== 107 && second !== 109 && second !== 115 && second !== 103) {
							type = 5;
						}

						close = -1;
						special++;
					}
				}
				// property/selector
				else {
					// { character, selector declaration
					if (code === 123) {
						depth++;

						// push flat css
						if (levels === 1 && flat.length !== 0) {
							levels = 0;
							flat = prefix + ' {' + flat + '}';

							// middleware, flat context
							if (uses === true) {
								temp = middleware(4, flat, line, column, prefix, output.length);

								if (temp != null) {
									flat = temp;
								}
							}

							output += flat;
							flat = '';
						}

						// nested selector
						if (depth === 2) {
							// discard first character {
							caret++;
							column++;

							// inner content of block
							inner = '';

							var nestSelector = buff.substring(0, buff.length - 1).split(splitPattern);
							var prevSelector = prev.substring(0, prev.length - 1).split(splitPattern);

							// keep track of opening `{` and `}` occurrences
							closed = 1;

							// travel to the end of the block
							while (caret < eof) {
								char = styles.charCodeAt(caret);

								// {, nested blocks may have nested blocks
								if (char === 123) {
									closed++;
								}
								// },
								else if (char === 125) {
									closed--;
								}

								// break when the nested block has ended
								if (closed === 0) {
									break;
								}

								// build content of nested block
								inner += styles.charAt(caret++);

								// move column and line position
								column = (char === 13 || char === 10) ? (line++, 0) : column + 1;
							}

							// handle multiple selectors: h1, h2 { div, h4 {} } should generate
							// -> h1 div, h2 div, h2 h4, h2 div {}
							length = prevSelector.length;

							for (var j = 0; j < length; j++) {
								// extract value, prep index for reuse
								temp = prevSelector[j];
								indexOf = temp.indexOf(prefix);

								prevSelector[j] = '';

								// since there could also be multiple nested selectors
								for (var k = 0, l = nestSelector.length; k < l; k++) {
									if (indexOf > 0) {
										selector = ':global(%)' + temp.trim();
									}
									else {
										selector = temp.replace(prefix, '&').trim();
									}

									sel = nestSelector[k].trim();

									if (sel.indexOf(' &') > 0) {
										selector = sel.replace('&', '').trim() + ' ' + selector;
									}
									else if (globalPattern.exec(sel) !== null) {
										selector = sel;
									}
									else {
										selector = selector + ' ' + sel;
									}

									prevSelector[j] += selector.replace(andSpacePattern, '').trim() + (k === l - 1  ? '' : ',');
								}
							}

							if (nest === undefined) {
								nest = '';
							}

							// concat nest
							nest += (
								'\n' +
								prevSelector.join(',').replace(globalsPattern, ' $1') +
								' {'+inner+'}'
							);

							// signature
							nested = 1;

							// clear current line, to avoid adding nested blocks to the normal flow
							buff = '';

							// decreament depth
							depth--;
						}
						// top-level selector
						else if (special === 0 || type === 2) {
							// register block with placeholder selector
							if (isplace === 0 && buff.indexOf('::place') !== -1) {
								isplace = 1;
							}

							selectors = buff.split(splitPattern);

							// current selector
							build = '';

							// previous selector
							prev = '';

							length = selectors.length;

							// prefix multiple selectors with namesapces
							// @example h1, h2, h3 --> [namespace] h1, [namespace] h1, ....
							for (var j = 0; j < length; j++) {
								char = (selector = selectors[j]).charCodeAt(0);

								// ` `, trim if first character is a space
								if (char === 32) {
									char = (selector = selector.trim()).charCodeAt(0);
								}

								// &
								if (char === 38) {
									// before: & { / &&... {
									selector = prefix + selector.substring(1).replace(andPattern, prefix);
									// after: ${prefix} { / ${prefix}${prefix}...
								}
								else {
									// default to :global if & exists outside of the first non-space character
									if ((indexOf = selector.indexOf(' &')) > 0) {
										// before: html & {
										selector = selector.replace(andPattern, prefix).trim();
										// after: html ${prefix} {
									}
									// :
									else if (char === 58) {
										nextcode = selector.charCodeAt(1);

										// h, t, :host
										if (compact === true && nextcode === 104 && selector.charCodeAt(4) === 116) {
											nextcode = selector.charCodeAt(5);

											// (, :host(selector)
											if (nextcode === 40) {
												// before: `(selector)`
												selector = (
													prefix + (
														selector
															.replace(hostPattern, '$1')
															.replace(andPattern, prefix)
													)
												);
												// after: ${prefx} selector {
											}
											// -, :host-context(selector)
											else if (nextcode === 45) {
												// before: `-context(selector)`
												selector = selector
													.replace(hostContextPattern, '$1 ' + prefix)
													.replace(andPattern, prefix)
												// after: selector ${prefix} {
											}
											// :host
											else {
												selector = prefix + selector.substring(5);
											}
										}
										// g, :global(selector)
										else if (
											nextcode === 103 &&
											(
												compact === true
												||
												((nextcode = selector.charCodeAt(8)) === 37)
											)
										) {
											globs = 1;

											// before: `:global(selector)`
											selector = (
												selector
													.replace(globalPattern, '$1')
													.replace(andPattern, prefix).trim()
											);
											// after: selector
										}
										// :hover, :active, :focus, etc...
										else {
											selector = prefix + selector;
										}
									}
									// non-pseudo selectors
									else if (globs === 0) {
										selector = prefix + ' ' + selector;
									}
								}

								// middleware, post-processed selector context
								if (uses === true) {
									temp = middleware(
										1.5,
										j === length - 1 ? selector.substring(0, selector.length - 1).trim() : selector,
										line,
										column,
										prefix,
										output.length
									);

									if (temp != null) {
										selector = j === length - 1 ? temp + ' {' : temp;
									}
								}

								// if first selector do not prefix with `,`
								prev += (j !== 0 ? ',\n' : '') + (globs !== 1 ? selector : ':global(%)' + selector);
								build += j !== 0 ? ',' + selector : selector;

								// reset :global flag
								globs = 0;
							}

							buff = build;
						}
						else {
							prev = buff;
						}
					}
					// not single `}`
					else if ((code === 125 && buff.length === 1) === false) {
						if (join === 1) {
							buff = buff.replace(newLinePattern, '');
						}

						// ;
						if (code !== 59) {
							buff = (code === 125 ? buff.substring(0, buff.length - 1) : buff.trim()) + ';';
						}

						// animation: a, n, i characters
						if (first === 97 && second === 110 && third === 105) {
							// removes ;
							buff = buff.substring(0, buff.length - 1);

							// position of :
							colon = buff.indexOf(':')+1;

							// left hand side everything before `:`
							build = buff.substring(0, colon);

							// short hand animation syntax
							if (animations === true && buff.charCodeAt(9) !== 45) {
								var anims = buff.substring(colon).trim().split(',');

								length = anims.length;

								// because we can have multiple animations `animation: slide 4s, slideOut 2s`
								for (var j = 0; j < length; j++) {
									var anim = anims[j];
									var props = anim.split(' ');

									// since we can't be sure of the position of the name of the animation we have to find it
									for (var k = 0, l = props.length; k < l; k++) {
										var prop = props[k].trim();
										var frst = prop.charCodeAt(0);
										var thrd = prop.charCodeAt(2);
										var len = prop.length;
										var last = prop.charCodeAt(len - 1);

										// animation name parser
										if (
											// first character
											(
											    // letters
											    (frst > 64 && frst < 90) ||
											    (frst > 96 && frst < 122) ||
											    // the exception `underscores or dashes`
											    frst === 45 ||
											    // but two dashes at the beginning are forbidden
											    (frst === 95 && prop.charCodeAt(1) !== 95)
											) &&

											// cubic-bezier()/steps(), )
											last !== 41 && len !== 0 &&

											!(
												frst === 105 &&
												(
													// infinite, i, f, e
													(thrd === 102 && last === 101 && len === 8) ||
													// initial
													(thrd === 105 && last === 108 && len === 7) ||
													// inherit
													(thrd === 104 && last === 116 && len === 7)
												)
											) &&

											// unset
											!(frst === 117 && thrd === 115 && last === 116 && len === 5) &&

											// linear, l, n, r
											!(frst === 108 && thrd === 110 && last === 114 && len === 6) &&

											// alternate/alternate-reverse, a, t, e
											!(frst === 97 && thrd === 116 && last === 101 && (len === 9 || len === 17)) &&

											// normal, n, r, l
											!(frst === 110 && thrd === 114 && last === 108 && len === 6) &&

											// backwards, b, c, s
											!(frst === 98 && thrd === 99 && last === 115 && len === 9) &&

											// forwards, f, r, s
											!(frst === 102 && thrd === 114 && last === 115 && len === 8) &&

											// both, b, t, h
											!(frst === 98 && thrd === 116 && last === 104 && len === 4) &&

											// none, n, n, e
											!(frst === 110 && thrd === 110 && last === 101 && len === 4)&&

											// running, r, n, g
											!(frst === 114 && thrd === 110 && last === 103 && len === 7) &&

											// paused, p, u, d
											!(frst === 112 && thrd === 117 && last === 100 && len === 6) &&

											// reversed, r, v, d
											!(frst === 114 && thrd === 118 && last === 100 && len === 8) &&

											// step-start/step-end, s, e, (t/d)
											!(
												frst === 115 && thrd === 101 &&
												((last === 116 && len === 10) || (last === 100 && len === 8))
											) &&

											// ease/ease-in/ease-out/ease-in-out, e, s, e
											!(
												frst === 101 && thrd === 115 &&
												(
													(last === 101 && len === 4) ||
													(len === 11 || len === 7 || len === 8) && prop.charCodeAt(4) === 45
												)
											) &&

											// durations, 0.4ms, .4s, 400ms ...
											isNaN(parseFloat(prop)) &&

											// handle spaces in cubic-bezier()/steps() functions
											prop.indexOf('(') === -1
										) {
											props[k] = animns + prop;
										}
									}

									build += (j === 0 ? '' : ',') + props.join(' ').trim();
								}
							}
							// explicit syntax, anims array should have only one element
							else {
								build += (
									(buff.charCodeAt(10) !== 110 ? '' : animns) +
									buff.substring(colon).trim().trim()
								);
							}

							// vendor prefix
							buff = webkit + build + ';' + build + (code === 125 ? ';}' : ';');
						}
						// appearance: a, p, p
						else if (first === 97 && second === 112 && third === 112) {
							// vendor prefix -webkit- and -moz-
							buff = (
								webkit + buff +
								moz + buff +
								buff
							);
						}
						// display: d, i, s
						else if (first === 100 && second === 105 && third === 115) {
							// flex/inline-flex
							if ((indexOf = buff.indexOf('flex')) !== -1) {
								// e, inline-flex
								temp = buff.charCodeAt(indexOf-2) === 101 ? 'inline-' : '';
								buff = buff.indexOf(' !important') !== -1 ? ' !important' : '';

								// vendor prefix
								buff = (
									'display: ' + webkit + temp + 'box' + buff + ';' +
									'display: ' + webkit + temp + 'flex' + buff + ';' +
									'display: ' + ms + 'flexbox' + buff + ';' +
									'display: ' + temp + 'flex' + buff + ';'
								);
							}
						}
						// transforms & transitions: t, r, a
						// text-size-adjust: t, e, x
						else if (
							first === 116 &&
							(
								(second === 114 && third === 97) ||
								(second === 101 && third === 120)
							)
						) {
							// vendor prefix -webkit- and -ms- if transform
							buff = (
								webkit + buff +
								(buff.charCodeAt(5) === 102 ? ms + buff : '') +
								buff
							);

							if (second + third === 211 && buff.charCodeAt(12) === 115 && buff.indexOf(' transform') > -1) {
								buff = buff.substring(0, buff.indexOf(';')+1).replace(transformPattern, ' '+webkit+'$1') + buff;
							}
						}
						// hyphens: h, y, p
						// user-select: u, s, e
						else if (
							(first === 104 && second === 121 && third === 112) ||
							(first === 117 && second === 115 && third === 101)
						) {
							// vendor prefix all
							buff = (
								webkit + buff +
								moz + buff +
								ms + buff +
								buff
							);
						}
						// flex: f, l, e
						else if (first === 102 && second === 108 && third === 101) {
							// vendor prefix all but moz
							buff = (
								webkit + buff +
								ms + buff +
								buff
							);
						}
						// order: o, r, d
						else if (first === 111 && second === 114 && third === 100) {
							// vendor prefix all but moz
							buff = (
								webkit + buff +
								ms + 'flex-' + buff +
								buff
							);
						}
						// align-items, align-center, align-self: a, l, i, -
						else if (first === 97 && second === 108 && third === 105 && buff.charCodeAt(5) === 45) {
							switch (buff.charCodeAt(6)) {
								// align-items, i
								case 105: {
									temp = buff.replace('-items', '');
									buff = (
										webkit + buff +
										webkit + 'box-' + temp +
										ms + 'flex-'+ temp +
										buff
									);
									break;
								}
								// align-self, s
								case 115: {
									buff = (
										ms + 'flex-item-' + buff.replace('-self', '') +
										buff
									);
									break;
								}
								// align-content
								default: {
									buff = (
										ms + 'flex-line-pack' + buff.replace('align-content', '') +
										buff
									);
									break;
								}
							}
						}
						// justify-content, j, u, s
						else if (first === 106 && second === 117 && third === 115) {
							colon = buff.indexOf(':');
							temp = buff.substring(colon).replace('flex-', '')

							buff = (
								webkit + 'box-pack' + temp +
								webkit + buff +
								ms + 'flex-pack' + temp +
								buff
							);
						}
						// cursor, c, u, r
						else if (first === 99 && second === 117 && third === 114 && /zoo|gra/.exec(buff) !== null) {
							buff = (
								buff.replace(colonPattern, ': ' + webkit) +
								buff.replace(colonPattern, ': ' + moz) +
								buff
							);
						}
						// width: min-content / width: max-content
						else if (first === 119 && second === 105 && third === 100 && (indexOf = buff.indexOf('-content')) !== -1) {
							temp = buff.substring(indexOf - 3);

							// vendor prefix
							buff = (
								'width: ' + webkit + temp +
								'width: ' + moz + temp +
								'width: ' + temp
							);
						}

						if (code !== 59) {
							buff = buff.substring(0, buff.length - 1);

							// }
							if (code === 125) {
								buff += '}';
							}
						}
					}

					// } character
					if (code === 125) {
						if (depth !== 0) {
							depth--;
						}

						// concat nested css
						if (depth === 0 && nested === 1) {
							styles = styles.substring(0, caret + 1) + nest + styles.substring(caret + 1);
							eof += nest.length;
							nest = '';
							nested = 0;
							close++;
						}

						// }, ` ` whitespace
						if (first !== 125 && buff.charCodeAt(buff.length - 2) === 32) {
							buff = buff.substring(0, buff.length-1).trim() + '}';
						}
					}

					// @keyframes
					if (special !== 0) {
						// }, find closing tag
						if (code === 125) {
							close++;
						}
						// {
						else if (code === 123 && close !== 0) {
							close--;
						}

						// closing tag
						if (close === 0) {
							// @keyframes
							if (type === 1) {
								// vendor prefix
								buff = '}@'+blob+'}';

								// reset
								blob = '';
							}

							// reset signatures
							type = 0;
							close--;
							special--;
						}
						// @keyframes
						else if (type === 1) {
							blob += buff;
						}
					}
					// flat context
					else if (depth === 0 && code !== 125) {
						levels = 1;
						flat = flat === undefined ? buff : flat + buff;
						buff = '';
					}
				}

				// append line to blck buffer
				blck += buff;

				// add blck buffer to output
				if (code === 125 && type === 0) {
					char = blck.charCodeAt(blck.length - 2);

					// {, @
					if (char !== 123) {
						// middleware, block context
						if (uses === true) {
							temp = middleware(3, blck, line, column, prefix, output.length);

							if (temp != null) {
								blck = temp;
							}
						}

						if (isplace === 1) {
							isplace = 0;

							blck = (
								blck.replace(placeholderPattern, '::'+webkit+'input-place') +
								blck.replace(placeholderPattern, '::'+moz+'place') +
								blck.replace(placeholderPattern, ':'+ms+'input-place') +
								blck
							);
						}

						// append blck buffer
						output += blck;
					}

					// reset blck buffer
					blck = '';
				}

				if (medias === 1) {
					// middleware, block context
					if (uses === true) {
						temp = middleware(3, blck, line, column, prefix, output.length);

						if (temp != null) {
							media = temp;
						}
					}

					output += media;
					medias = 0;
					media = '';
				}

				join = 0

				// reset line buffer
				buff = '';
			}
			// build line by line
			else {
				// \r, \n, new lines
				if (code === 13 || code === 10) {
					if (comline === 1) {
						comment = comline = 0;
						buff = buff.substring(0, buff.indexOf('//')).trim();
					}
					// /
					else if (uses === true && comment === 0 && (length = (str = str.trim()).length) !== 0 && str.charCodeAt(0) !== 47) {
						if (buff.length !== 0) {
							temp = middleware(7, str, line, column, prefix, output.length);

							if (temp != null) {
								buff = buff.replace(new RegExp(str+'$'), temp).trim();
							}
						}

						str = '';
					}

					column = 0;
					line++;
				}
				else {
					// not `\t` tab character
					if (code !== 9) {
						character = styles.charAt(caret);

						// build line buffer
						if (uses === true && comment === 0) {
							str += character;
						}

						// build character buffer
						buff += character;

						switch (code) {
							// ,
							case 44: {
								if (strings === 0 && comment === 0 && func === 0) {
									join = 1;
									buff += '\n';
								}
								break;
							}
							// " character
							case 34: {
								if (comment === 0) {
									// exit string " context / enter string context
									strings = strings === 34 ? 0 : (strings === 39 ? 39 : 34);
								}
								break;
							}
							// ' character
							case 39: {
								if (comment === 0) {
									// exit string ' context / enter string context
									strings = strings === 39 ? 0 : (strings === 34 ? 34 : 39);
								}
								break;
							}
							// ( character
							case 40: {
								if (strings === 0 && comment === 0) {
									func = 1;
								}
								break;
							}
							// ) character
							case 41: {
								if (strings === 0 && comment === 0) {
									func = 0;
								}
								break;
							}
							// / character
							case 47: {
								if (strings === 0 && func === 0) {
									char = styles.charCodeAt(caret - 1);

									// /, begin line comment
									if (comblck === 0 && char === 47) {
										comment = comline = 1;
									}
									// *, end block comment
									else if (char === 42) {
										comment = comblck = 0;
										buff = buff.substring(0, buff.indexOf('/*')).trim();
									}
								}

								break;
							}
							// * character
							case 42: {
								if (strings === 0 && func === 0 && comline === 0 && comblck === 0) {
									// /, begin block comment
									if (styles.charCodeAt(caret - 1) === 47) {
										comment = comblck = 1;
									}
								}

								break;
							}
						}
					}

					// move column position
					column++;
				}
			}

			// move caret position
			caret++;
		}

		// trailing flat css
		if (flat !== undefined && flat.length !== 0) {
			flat = prefix + ' {' + flat + '}';

			// middleware, flat context
			if (uses === true) {
				temp = middleware(4, flat, line, column, prefix, output.length);

				if (temp != null) {
					flat = temp;
				}
			}

			// append flat css
			output += flat;
		}

		// middleware, output context
		if (uses === true) {
			temp = middleware(6, output, line, column, prefix, output.length);

			if (temp != null) {
				output = temp;
			}
		}

		return output;
	}


	/**
	 * use plugin
	 *
	 * @param  {string|function|function[]} key
	 * @param  {function?} plugin
	 * @return {Object} {plugins}
	 */
	function use (plugin) {
		var length = plugins.length;

		if (plugin != null) {
			// array of plugins
			if (plugin.constructor === Array) {
				for (var i = 0, l = plugin.length; i < l; i++) {
					plugins[length++] = plugin[i];
				}
			}
			// single un-keyed plugin
			else {
				plugins[length] = plugin;
			}
		}

		return stylis;
	};

	/**
	 * Middleware Proxy
	 *
	 * @param  {Number} ctx
	 * @param  {String} str
	 * @param  {Number} line
	 * @param  {Number} col
	 * @param  {String} prefix
	 * @param  {Number} length
	 * @return {String?}
	 */
	function proxy (ctx, str, line, col, prefix, length) {
		var output = str;

		for (var i = 0, l = plugins.length; i < l; i++) {
			output = plugins[i](ctx, output, line, col, prefix, length) || output;
		}

		if (output !== str) {
			return output;
		}
	}

	stylis.use = use;


	/**
	 * plugin store
	 *
	 * @type {Function[]}
	 */
	stylis.p = plugins;


	/**
	 * regular expresions
	 *
	 * @type {Object<string, RegExp>}
	 */
	stylis.r = {
		a: andPattern,
		s: splitPattern,
		g: globalPattern,
		n: globalsPattern
	};


	return stylis;
}));
