'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.eliminateGlobals = exports.getParentSelectorClassesMap = exports.getExtendClassesMap = exports.getComposesClassesMap = exports.getRegularClassesMap = exports.getICSSExportPropsMap = undefined;

var _fp = require('lodash/fp');

var _fp2 = _interopRequireDefault(_fp);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var getICSSExportPropsMap = exports.getICSSExportPropsMap = function getICSSExportPropsMap(ast) {
  var ruleSets = [];
  ast.traverseByType('ruleset', function (node) {
    return ruleSets.push(node);
  });

  return _fp2.default.compose(_fp2.default.reduce(function (result, key) {
    var prop = _fp2.default.compose(_fp2.default.nth(0), _fp2.default.map('content'))(key);
    result[prop] = prop; // e.g. { myProp: 'myProp' }
    return result;
  }, {}), _fp2.default.map('content'), _fp2.default.filter({ type: 'property' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'declaration' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'block' }), _fp2.default.flatMap('content'), _fp2.default.filter({
    content: [{
      type: 'selector',
      content: [{
        type: 'pseudoClass',
        content: [{
          type: 'ident', content: 'export'
        }]
      }]
    }]
  }))(ruleSets);
};
/* eslint-disable no-param-reassign */
var getRegularClassesMap = exports.getRegularClassesMap = function getRegularClassesMap(ast) {
  var ruleSets = [];
  ast.traverseByType('ruleset', function (node) {
    return ruleSets.push(node);
  });

  return _fp2.default.compose(_fp2.default.reduce(function (result, key) {
    result[key] = false; // classes haven't been used
    return result;
  }, {}), _fp2.default.map('content'), _fp2.default.filter({ type: 'ident' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'class' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'selector' }), _fp2.default.flatMap('content'))(ruleSets);
};

var getComposesClassesMap = exports.getComposesClassesMap = function getComposesClassesMap(ast) {
  var declarations = [];
  ast.traverseByType('declaration', function (node) {
    return declarations.push(node);
  });

  return _fp2.default.compose(_fp2.default.reduce(function (result, key) {
    result[key] = true; // mark composed classes as true
    return result;
  }, {}), _fp2.default.flatMap(_fp2.default.compose(_fp2.default.map(_fp2.default.get('content')), _fp2.default.filter({ type: 'ident' }), _fp2.default.get('content'), _fp2.default.find({ type: 'value' }), _fp2.default.get('content'))),
  /*
     reject classes composing from other files
     eg.
     .foo {
     composes: .bar from './otherFile';
     }
   */
  _fp2.default.reject(_fp2.default.compose(_fp2.default.find({ type: 'ident', content: 'from' }), _fp2.default.get('content'), _fp2.default.find({ type: 'value' }), _fp2.default.get('content'))), _fp2.default.filter(_fp2.default.compose(_fp2.default.find({ type: 'ident', content: 'composes' }), _fp2.default.get('content'), _fp2.default.find({ type: 'property' }), _fp2.default.get('content'))))(declarations);
};

var getExtendClassesMap = exports.getExtendClassesMap = function getExtendClassesMap(ast) {
  var extendNodes = [];
  ast.traverseByType('extend', function (node) {
    return extendNodes.push(node);
  });

  return _fp2.default.compose(_fp2.default.reduce(function (result, key) {
    result[key] = true; // mark extend classes as true
    return result;
  }, {}), _fp2.default.map(_fp2.default.compose(_fp2.default.get('content'), _fp2.default.find({ type: 'ident' }), _fp2.default.get('content'), _fp2.default.find({ type: 'class' }), _fp2.default.get('content'), _fp2.default.find({ type: 'selector' }), _fp2.default.get('content'))))(extendNodes);
};

/**
 * Resolves parent selectors to their full class names.
 *
 * E.g. `.foo { &_bar {color: blue } }` to `.foo_bar`.
 */
var getParentSelectorClassesMap = exports.getParentSelectorClassesMap = function getParentSelectorClassesMap(ast) {
  var classesMap = {};

  // Recursively traverses down the tree looking for parent selector
  // extensions. Recursion is necessary as these selectors can be nested.
  var getExtensions = function getExtensions(nodeContent) {
    var blockContent = _fp2.default.compose(_fp2.default.flatMap('content'), _fp2.default.filter({ type: 'block' }))(nodeContent);

    var rulesetsContent = _fp2.default.flatMap('content', _fp2.default.concat(
    // `ruleset` children
    _fp2.default.filter({ type: 'ruleset' }, blockContent),

    // `ruleset` descendants nested in `include` nodes
    _fp2.default.compose(_fp2.default.filter({ type: 'ruleset' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'block' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'include' }))(blockContent)));

    var extensions = _fp2.default.compose(_fp2.default.map('content'), _fp2.default.filter({ type: 'ident' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'parentSelectorExtension' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'selector' }))(rulesetsContent);

    if (!extensions.length) return [];

    var nestedExtensions = getExtensions(rulesetsContent);
    var result = extensions;
    if (nestedExtensions.length) {
      nestedExtensions.forEach(function (nestedExt) {
        extensions.forEach(function (ext) {
          result.push(ext + nestedExt);
        });
      });
    }

    return result;
  };

  ast.traverseByType('ruleset', function (node) {
    var classNames = _fp2.default.compose(_fp2.default.map('content'), _fp2.default.filter({ type: 'ident' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'class' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'selector' }))(node.content);

    if (!classNames.length) return;

    var extensions = getExtensions(node.content);
    if (!extensions.length) return;

    classNames.forEach(function (className) {
      extensions.forEach(function (ext) {
        classesMap[className + ext] = false;
      });

      // Ignore the base class if it only exists for nesting parent selectors
      var hasDeclarations = _fp2.default.compose(_fp2.default.filter({ type: 'declaration' }), _fp2.default.flatMap('content'), _fp2.default.filter({ type: 'block' }))(node.content).length > 0;
      if (!hasDeclarations) classesMap[className] = true;
    });
  });

  return classesMap;
};

/*
   mutates ast by removing instances of :global
 */
var eliminateGlobals = exports.eliminateGlobals = function eliminateGlobals(ast) {
  ast.traverse(function (node, index, parent) {
    if (node.type === 'ruleset') {
      if (_fp2.default.compose(_fp2.default.negate(_fp2.default.isEmpty), _fp2.default.find({ type: 'ident', content: 'global' }), _fp2.default.get('content'), _fp2.default.find({ type: 'pseudoClass' }), _fp2.default.get('content'), _fp2.default.find({ type: 'selector' }), _fp2.default.get('content'))(node)) {
        parent.removeChild(index);
      }
    }
  });
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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